param (
    [string]$jre_home,
    [string]$application_home,
    [string]$service_name
)

if (-not $jre_home -or -not $application_home -or -not $service_name) {
    Write-Host "ERROR: Missing required parameters."
    Write-Host "Usage: .\script.ps1 -jre_home <path_to_jre> -application_home <path_to_application> -service_name <short name>"
    exit 1
}

function Timestamp { Get-Date -Format "yyyy-MM-dd HH:mm:ss" }

$current_directory = $PSScriptRoot
$customer_home = [IO.Path]::GetFullPath([IO.Path]::Combine($current_directory, ".."))
$config_file = [IO.Path]::GetFullPath([IO.Path]::Combine($customer_home, "config", "config.json"))
$service_config_file = [IO.Path]::GetFullPath([IO.Path]::Combine($customer_home, "config", "service.json"))
$log4j_file = [IO.Path]::GetFullPath([IO.Path]::Combine($customer_home, "config", "log4j2.xml"))
$logs_dir = [IO.Path]::GetFullPath([IO.Path]::Combine($customer_home, "logs"))
$service_exec = [IO.Path]::GetFullPath([IO.Path]::Combine($current_directory, "fortifiedid_access_srv.exe"))
$jre_dll = [IO.Path]::GetFullPath([IO.Path]::Combine($jre_home, "bin", "server", "jvm.dll"))

$assignedMemory = 1024
$threadStackSize = 512
$application_display_name = "Fortified ID Access"
$service_description = "Fortified ID Access is an Identity Provider that manages authentication, authorization controls, and single sign-on functionality."
$service_std_out = [IO.Path]::GetFullPath([IO.Path]::Combine($logs_dir, "service-stdout.log"))
$service_std_error = [IO.Path]::GetFullPath([IO.Path]::Combine($logs_dir, "service-stderror.log"))

# Build JAVA_OPTS
$JAVA_OPTS = "-Dlog4j.configurationFile=$log4j_file;"
$JAVA_OPTS += "-Dorg.apache.xml.security.ignoreLineBreaks=true;"
$JAVA_OPTS += "-Dfile.encoding=UTF-8;"
$JAVA_OPTS += "-Dfoss.platform.server.logEnabled=true;"
$JAVA_OPTS += "-Dfoss.platform.server.stdoutEnabled=true;"
$JAVA_OPTS += "-Dcustomer_home=$customer_home;"
$JAVA_OPTS += "-Dfoss.commons.logging.sanitation.keywords=password,passcode,pwd,unicodepwd,userpassword,otp,secret,pin,bind_password,http_key_password,ssl_key_password,ssl_keystore_password,ssl_truststore_password,sign_ref_keystore_password,sign_ref_keystore_key_password,key_password,sign_jwt_keystore_password,encryption_key,new_password;"

if (Test-Path $service_config_file) {
    $configData = Get-Content -Path $service_config_file -Raw | ConvertFrom-Json
    if ($configData.java_opts) {
        foreach ($option in $configData.java_opts) {
            if ($option.key -and $option.value) {
                $JAVA_OPTS += "-$($option.key)=$($option.value);"
            }
        }
    }
    if ($configData.assignedMemory) { $assignedMemory = $configData.assignedMemory }
    if ($configData.threadStackSize) { $threadStackSize = $configData.threadStackSize }
}
else {
    Write-Host "$(Timestamp) WARNING: Service configuration file not found: $service_config_file"
}

$newClasspath = "$application_home\lib\*;"

# Procrun 2.0 (64-bit) registry path
$javaRegPath = "HKLM:\SOFTWARE\Wow6432Node\Apache Software Foundation\Procrun 2.0\$service_name\Parameters\Java"
$startRegPath = "HKLM:\SOFTWARE\Wow6432Node\Apache Software Foundation\Procrun 2.0\$service_name\Parameters\Start"
$stopRegPath = "HKLM:\SOFTWARE\Wow6432Node\Apache Software Foundation\Procrun 2.0\$service_name\Parameters\Stop"

# --- If service exists, update only Classpath ---
if (Test-Path -LiteralPath $javaRegPath) {
    $service = Get-Service -Name $service_name -ErrorAction SilentlyContinue
    if (-not $service) {
        Write-Host "$(Timestamp) ERROR: Service '$service_name' exists in registry but is not registered as a Windows service."
        exit 1
    }

    $wasRunning = $service.Status -eq 'Running'
    if ($wasRunning) {
        Write-Host "$(Timestamp) Stopping service '$service_name'..."
        Stop-Service -Name $service_name -Force
        $service.WaitForStatus('Stopped', '00:00:20') | Out-Null
        Write-Host "$(Timestamp) Service stopped."
    }

    Write-Host "$(Timestamp) Updating Classpath in registry..."
    $oldValue = (Get-ItemProperty -Path $javaRegPath -Name 'Classpath' -ErrorAction SilentlyContinue).Classpath
    Write-Host "$(Timestamp) Old Classpath: $oldValue"
    Set-ItemProperty -Path $javaRegPath -Name "Classpath" -Value $newClasspath -Type String
    Set-ItemProperty -Path $startRegPath -Name "WorkingPath" -Value $application_home -Type String
    Set-ItemProperty -Path $stopRegPath  -Name "WorkingPath" -Value $application_home -Type String
    Write-Host "$(Timestamp) New Classpath: $newClasspath"
    Write-Host "$(Timestamp) Classpath updated successfully."

    if ($wasRunning) {
        Write-Host "$(Timestamp) Starting service '$service_name'..."
        Start-Service -Name $service_name
        Write-Host "$(Timestamp) Service started."
    }

    Write-Host "$(Timestamp) Update process completed."
    exit 0
}

# --- Else: full installation (original behaviour) ---
Write-Host "$(Timestamp) Service '$service_name' not found in Procrun registry. Proceeding with full installation..."

$service_params = "--DisplayName=`"$application_display_name`""
$service_params += " --Description=`"$service_description`""
$service_params += " --StartClass=`"foss.platform.server.Server`""
$service_params += " --StopClass=`"foss.platform.server.Server`""
$service_params += " --StartMethod=main"
$service_params += " --JvmOptions=`"$JAVA_OPTS`""
$service_params += " --Classpath=`"$newClasspath`""
$service_params += " --Jvm=`"$jre_dll`""
$service_params += " --JavaHome=`"$jre_home`""
$service_params += " ++Environment=`"SERVER_LOG_DIR=$logs_dir;`""
$service_params += " --StartMode=jvm"
$service_params += " --StopMode=jvm"
$service_params += " --StartPath=`"$application_home`""
$service_params += " --StopPath=`"$application_home`""
$service_params += " ++StartParams=`"Node;$config_file`""
$service_params += " --StopMethod=stop"
$service_params += " --StopTimeout=5"
$service_params += " --Startup=auto"
$service_params += " --StdOutput=`"$service_std_out`""
$service_params += " --StdError=`"$service_std_error`""
$service_params += " --LogPath=`"$logs_dir`""
$service_params += " --PidFile=`"process.pid`""
$service_params += " --ServiceUser=LocalSystem"
$service_params += " --JvmMs=$assignedMemory"
$service_params += " --JvmMx=$assignedMemory"
$service_params += " --JvmSs=$threadStackSize"

$cmd = "& '$service_exec' //IS//$service_name $service_params"
Write-Host "$(Timestamp) Installing service with command:"
Write-Host "  $cmd"
Invoke-Expression $cmd
Write-Host "$(Timestamp) Installation finished."