<?php declare(strict_types=1);

namespace B2B\CustomerCacheIndex\Subscriber;

use Shopware\Core\PlatformRequest;
use Shopware\Core\System\SalesChannel\SalesChannelContext;
use Shopware\Storefront\Framework\Routing\MaintenanceModeResolver;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

class CacheKeySubscriber implements EventSubscriberInterface
{

	public function __construct(
		private readonly bool $httpCacheEnabled,
		private readonly MaintenanceModeResolver $maintenanceResolver,
	)
	{
	}
	public static function getSubscribedEvents(): array
	{
		return [
			KernelEvents::RESPONSE => [['updateHttpCacheHash', -1600]]
		];
	}

	public function updateHttpCacheHash(ResponseEvent $event): void
	{
		if (!$this->httpCacheEnabled) {
			return;
		}

		$request = $event->getRequest();

		if (!$this->maintenanceResolver->shouldBeCached($request)) {
			return;
		}

		$response = $event->getResponse();
		$context = $request->attributes->get(PlatformRequest::ATTRIBUTE_SALES_CHANNEL_CONTEXT_OBJECT);

		if (!$context instanceof SalesChannelContext) {
			return;
		}

		if ($context->getCustomer()) {
			$newCacheHash = $this->buildCacheHash($context);

			if ($request->cookies->get('sw-cache-hash', '') !== $newCacheHash) {
				$cookie = Cookie::create('sw-cache-hash', $newCacheHash);
				$cookie->setSecureDefault($request->isSecure());

				$response->headers->setCookie($cookie);
			}
		}
	}

	private function buildCacheHash(SalesChannelContext $context): string
	{
		return md5(json_encode(array_filter([
			$context->getRuleIds(),
			$context->getContext()->getVersionId(),
			$context->getCurrency()->getId(),
			$context->getTaxState(),
			$context->getCustomer() ? 'logged-in' : 'not-logged-in',
			$context->getCustomer() ? $context->getCustomerId() : null,
		]), \JSON_THROW_ON_ERROR));
	}
}